Scriptname JMCR_ModifierPerk extends Quest  
{Main script to handle cost modifiers and perks.}


Actor Property PlayerRef Auto
{Reference to the player character.}

float Property UpdateFrequency Auto
{If the system is busy and a modifier is updated, it will wait UpdateFrequency until trying again to apply the modifier.}

Perk Property ModifierPerk Auto
{Perk that modifies spell cost.}

String Property ModifierName Auto
{Name used in debug messages.}

String Property ActorValue Auto
{Actor value modified by the vanilla enchantment.}

int currentCheckNumber = 1

int Function GetNewCheckNumber()
	currentCheckNumber += 1
	return currentCheckNumber
EndFunction

; General Functions

Function Error(string m)
	Debug.TraceStack("BetterSpellCostStacking: [" + ModifierName + "] Error: " + m)
	Debug.Notification("BetterSpellCostStacking - Error:\n" + m)
EndFunction

Function DebugInfo(string m)
	Debug.Trace("BetterSpellCostStacking: [" + ModifierName + "]" + m)
EndFunction

Event OnUpdate()
	UpdatePerkModifier()
EndEvent

; Reapply perk modification
Function OnLoadSaveModifier()
	ModifierPerk.SetNthEntryValue(0, 0, modifierValue[0])
EndFunction

; Add perk
Function EnableModifier()
	PlayerRef.AddPerk(ModifierPerk)
EndFunction

; Remove perk and modifiers
Function DisableModifier()
	PlayerRef.RemovePerk(ModifierPerk)
	; Remove the first modifier, until no modifiers are left
	while nextModifier[0] != 0
		RemoveModifier(nextModifier[0], checkNumberModifier[nextModifier[0]])
	endwhile
EndFunction


; Modifier object (doubly linked list)
; Member Variables:
float [] modifierValue		; The modifier value.
int [] checkNumberModifier  ; Unique index used to verify an index belongs to the same object (indices are reused)
int [] nextModifier			; The next element in the list.
int [] prevModifier			; The previous element in the list.
; OOP Variables:
int [] recycleModifier		; Linked list of indices to recycle: recycle[0] -> recycle[recycle[0]] -> ...
int allocatedModifier		; Number of indices used

; Create arrays and set initial values
Function InitModifier()
	modifierValue 					= new float[128]
	modifierValue[0]				= 1.0
	checkNumberModifier				= new int[128]
	nextModifier						= new int[128]
	prevModifier						= new int[128]
	recycleModifier					= new int[128]
	recycleModifier[0]				= 0
	allocatedModifier					= 0
	nextModifier[0]					= 0
	prevModifier[0]					= 0
EndFunction

; Allocate new index for the modifier object
int Function AllocateModifier()
	if recycleModifier[0] == 0
		allocatedModifier += 1
		if allocatedModifier >= 128
			Error("Index out of bounds: " + allocatedModifier)
			return -1
		endif
		return allocatedModifier
	else
		int tmp = recycleModifier[0]
		recycleModifier[0] = recycleModifier[recycleModifier[0]]
		return tmp
	endif
EndFunction

; Deallocate index of the modifier object
Function DeallocateModifier(int index)
	int tmp = index
	recycleModifier[tmp] = recycleModifier[0]
	recycleModifier[0] = tmp
EndFunction

; Recalculate total modifier and update perk
Auto State readyState
	Function UpdatePerkModifier()
		GotoState("busyState")
		;DebugInfo("UpdatePerkModifier()")
		
		
		float tmp = 1.0
		;DebugInfo("tmp = " + tmp)
		int currentIndex = nextModifier[0]
		;DebugInfo("currentIndex = " + currentIndex)
		
		; Calculate total modifier
		while currentIndex != 0
			tmp *= modifierValue[currentIndex]
			;DebugInfo("index = " + currentIndex + " checkNumber = " + checkNumberModifier[currentIndex])
			;DebugInfo("tmp *= " + modifierValue[currentIndex])
			;DebugInfo("tmp = " + tmp)
			currentIndex = nextModifier[currentIndex]
		endwhile
		
		modifierValue[0] = tmp
		ModifierPerk.SetNthEntryValue(0, 0, tmp)
		
		GotoState("readyState")
	EndFunction
EndState

; Cannot update perk, delay for UpdateFrequency seconds
State busyState
	Function UpdatePerkModifier()
		RegisterForSingleUpdate(UpdateFrequency)
	EndFunction
EndState

Function UpdatePerkModifier()
	Error("Unexpected State")
EndFunction

; Get the check number for the modifier object
; Used directly after adding the modifier (only one value, the index can be returned)
int Function GetcheckNumberModifier(int index)
	return checkNumberModifier[index]
EndFunction

; Add new modifier
int Function AddModifier(float value)
	;DebugInfo("AddModifier(): Start")
	int index = AllocateModifier()
	;DebugInfo("index = " + index)
	if index == -1
		return -1
	endif
	
	; Old last element
	int oldLast = prevModifier[0]
	; Set new last element
	prevModifier[0] = index
	nextModifier[oldLast] = index
	
	nextModifier[index] = 0
	prevModifier[index] = oldLast
	
	modifierValue[index] = value

	float mod = (1.0 - modifierValue[index]) * 100.0
	PlayerRef.ModActorValue(ActorValue, -mod )
	; The checkNumber is used to recognize, if the modifier was removed and a new modifier created with the same index
	checkNumberModifier[index] = GetNewCheckNumber()
	;DebugInfo("modifierValue["+index+"] = " + modifierValue[index])
	;DebugInfo("checkNumberModifier["+index+"] = " + checkNumberModifier[index])
	UpdatePerkModifier()
	;DebugInfo("AddModifier(): Done")
	return index
EndFunction

; Remove a modifier
Function RemoveModifier(int index, int checkNumber)
	;DebugInfo("RemoveModifier(" + index + "," + checkNumber + "): Start")

	; Only remove the modifier, if the checkNumber is correct
	; If the checkNumber differs, the modifier was removed and a new modifier was created, that uses the same index
	if checkNumberModifier[index] != checkNumber
		return
	endif
	checkNumberModifier[index] = 0

	int prev = prevModifier[index]
	int next = nextModifier[index]

	nextModifier[prev] = next
	prevModifier[next] = prev

	float mod = (1.0 - modifierValue[index]) * 100.0
	PlayerRef.ModActorValue(ActorValue, mod )

	DeallocateModifier(index)
	UpdatePerkModifier()
	;DebugInfo("RemoveModifier(" + index + "," + checkNumber + "): Done")
EndFunction